#include "Gate_AND.h"

Gate_AND::Gate_AND(int gateID, gType type, int numInputs, int gateLevel)
    : Gate_BASE(gateID, type, numInputs, gateLevel)
{
    textRect = QRectF(BORDER_OFFSET, BORDER_OFFSET, xSize - BORDER_OFFSET * 3, ySize - BORDER_OFFSET * 2);
}

void Gate_AND::paint(QPainter *painter, const QStyleOptionGraphicsItem *option, QWidget *widget)
{
    Q_UNUSED(widget);
    Q_UNUSED(option);

#ifdef _DEBUG
    painter->save();
    painter->setPen((auxSelected) ? debugSelectedPen : debugPen);
    painter->setBrush(debugBrush);
    painter->drawRect(0, 0, xSize, ySize);
    painter->drawRect(textRect);
    if (numInputs < 2) {
        painter->setPen(debugErrorPen);
        painter->drawLine(0, 0, xSize, ySize);
        painter->drawLine(xSize, 0, 0, ySize);
    }
    painter->restore();
#endif

    if (auxSelected) painter->setPen(highlightedPen);
    else painter->setPen(defaultPen);
    painter->setBrush(defaultBrush);

    // Draw circles indicating I/O points
    for (int i = 0; i < numInputs; i++) {
        painter->drawEllipse(inputPoints[i], INPUT_CIRCLE_SIZE, INPUT_CIRCLE_SIZE);
    }
    painter->drawEllipse(outputPoint, INPUT_CIRCLE_SIZE, INPUT_CIRCLE_SIZE);

    // Draw gate outline
    painter->drawLine(BORDER_OFFSET, BORDER_OFFSET, BORDER_OFFSET, ySize - BORDER_OFFSET);
    painter->drawLine(BORDER_OFFSET, BORDER_OFFSET, xSize - (BORDER_OFFSET * (numInputs + 1.5)), BORDER_OFFSET);
    painter->drawLine(BORDER_OFFSET, ySize - BORDER_OFFSET, xSize - (BORDER_OFFSET * (numInputs + 1.5)), ySize - BORDER_OFFSET);

    QRect arcBox(xSize - BORDER_OFFSET * (numInputs + 1) * 2, BORDER_OFFSET, BORDER_OFFSET * (numInputs + 1) * 2 - BORDER_OFFSET, ySize - BORDER_OFFSET * 2);
    painter->drawArc(arcBox, -90 * 16, 180 * 16);

    // Draw I/O lines
    for (int i = 0; i < numInputs; i++) {
        painter->drawLine(inputPoints[i], QPointF(inputPoints[i].x() + BORDER_OFFSET, inputPoints[i].y()));
    }
    painter->drawLine(outputPoint, QPointF(outputPoint.x() - BORDER_OFFSET, outputPoint.y()));

    // Draw text showing gate ID
    painter->setPen(defaultPen);
    painter->setFont(defaultFont);
    painter->drawText(textRect, Qt::AlignCenter, QString::number(gateID));

    // If enqueued, draw circle around gate ID
    if (enqueued) {
        painter->drawEllipse(textRect.center(), ENQUEUED_CIRCLE_WIDTH, ENQUEUED_CIRCLE_WIDTH);
    }
}

void Gate_AND::simulateToOutput()
{
    // Save initial values to compare to later
    logicValue initValue = outputValue;
    logicValue initFaultyValue = outputFaultyValue;

    // Compute new output values
    bool undefined = false, undefinedFaulty = false;
    bool allOnes = true, allOnesFaulty = true;
    bool allZeros = true, allZerosFaulty = true;
    for (int i = 0; i < numInputs; i++) {
        if (inputValues[i] == logicValue_X) undefined = true;
        if (inputValues[i] == logicValue_0) allOnes = false;
        if (inputValues[i] == logicValue_1) allZeros = false;

        if (inputFaultyValues[i] == logicValue_X) undefinedFaulty = true;
        if (inputFaultyValues[i] == logicValue_0) allOnesFaulty = false;
        if (inputFaultyValues[i] == logicValue_1) allZerosFaulty = false;
    }

    if (undefined) outputValue = logicValue_X;
    else if (allOnes) outputValue = logicValue_1;
    else outputValue = logicValue_0;

    if (undefinedFaulty) outputFaultyValue = logicValue_X;
    else if (allOnesFaulty) outputFaultyValue = logicValue_1;
    else outputFaultyValue = logicValue_0;

    // If outputs have changed, queue the connected gate for simulation
    if (outputValue != initValue || outputFaultyValue != initFaultyValue)
        emit enqueueSim(this);

    // Update connected wire values
    for (int i = 0; i < gateOutputWires.size(); i++) {
        gateOutputWires[i]->setValue(outputValue, outputFaultyValue, false);
    }
}
